<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\PrintSetting;
use App\Models\Customer;
use App\Models\Bill;
use App\Models\BillItem;
use App\Models\OutsideItem;
use App\Models\CreditDetail;
use App\Models\CreditAdjustment;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
class BillController extends Controller
{
   public function index()
   {
    return view('bill.makeBill');
   }
   public function submitCart(Request $request)
{
    $validated = $request->validate([
        'products' => 'required|array',
        'products.*.product_id' => 'required|exists:products,id',
        'products.*.qty' => 'required|integer|min:1',
    ]);

    // Save to session
    session(['cart_data' => $validated['products']]);

    // Redirect to the bill.create route
    return redirect()->route('bill.create');
}

public function buyOutside($id)
{
    $product=Product::findOrfail($id);
    return view('bill.buyoutsideform' , compact('product'));
}
public function storeOutside(Request $request)
{
    $validated = $request->validate([
        'products' => 'required|array',
        'products.*.id' => 'required|exists:products,id',
        'products.*.qty' => 'required|numeric|min:1',
        'products.*.price' => 'required|numeric|min:0',
        'products.*.total' => 'required|numeric|min:0',
        'products.*.sold_as' => 'required|in:packet,loss_pack',
        'payment_type' => 'required|in:cash,credit',
        'remarks' => 'nullable|string',
        'cash_customer_name' => 'nullable|string',
        'customer_id' => 'nullable|exists:customers,id',
        'new_customer_name' => 'nullable|string',
        'new_customer_phone' => 'nullable|string',
        'new_customer_address' => 'nullable|string',
    ]);

    $customerId = null;
    $customerName = null;
    $grandTotal = 0;

    // Calculate grand total from all products
    foreach ($request->products as $productData) {
        $grandTotal += $productData['total'];
    }

    // Handle customer creation/selection for credit payments
    if ($request->payment_type === 'credit') {
        if ($request->filled('customer_id')) {
            $customer = Customer::find($request->customer_id);
            $customerId = $customer->id;
            $customerName = $customer->name;

            $creditDetail = CreditDetail::updateOrCreate(
                ['customer_id' => $customerId],
                ['due_amount' => \DB::raw("due_amount + $grandTotal")]
            );
        } elseif ($request->filled('new_customer_name') && $request->filled('new_customer_phone')) {
            $customer = Customer::create([
                'name' => $request->new_customer_name,
                'phone' => $request->new_customer_phone,
                'address' => $request->new_customer_address ?? null,
            ]);

            $customerId = $customer->id;
            $customerName = $customer->name;

            CreditDetail::create([
                'customer_id' => $customerId,
                'due_amount' => $grandTotal,
                'is_paid' => false,
            ]);
        } else {
            return redirect()->back()->withErrors(['customer' => 'Customer information is required for credit payment.']);
        }
    } else {
        $customerName = $request->cash_customer_name ?? 'Cash Customer';
    }

    // Create the bill
    $bill = Bill::create([
        'bill_number' => 'BILL-' . uniqid(),
        'customer_id' => $customerId,
        'customer_name' => $customerName,
        'print_type' => 'by_name', // Default value since not in form
        'remarks' => $request->remarks,
        'total_amount' => $grandTotal,
        'payment' => $request->payment_type,
        'price_type' => $request->products[0]['price_type'] ?? 'retail', // Get from first product
        'date' => now(),
    ]);

    // Store bill items
    foreach ($request->products as $productData) {
        $product = Product::find($productData['id']);

        if (!$product) {
            continue; // Skip if product not found (shouldn't happen due to validation)
        }

        BillItem::create([
            'bill_id' => $bill->id,
            'product_id' => $productData['id'],
            'sale' => 'outside', // From your hidden input
            'qty' => $productData['qty'],
            'price' => $productData['price'],
            'sold_as' => $productData['sold_as'],
            'retail_price' => $product->retail_price,
            'less_from_retail' => $productData['price'] - $product->retail_price,
            'amount' => $productData['total'],
        ]);
    }

    return redirect()->route('print.view', $bill->id)->with('success', 'Bill submitted successfully!');
}
public function showBillCreate()
{
    $cartItems = session('cart_data', []);

    // Fetch full product info and merge with qty
    $cartWithDetails = collect($cartItems)->map(function ($item) {
        $product = Product::find($item['product_id']);
        return [
            'product' => $product,
            'qty' => $item['qty'],
        ];
    });

    return view('bill.create', compact('cartWithDetails'));
}
public function BillStore(Request $request)
{
   // dd($request->all());
    $validated = $request->validate([
        'products' => 'required|array',
        'print_type' => 'required|string',
        'remarks' => 'nullable|string',
        'grand_total' => 'required|string',
        'sale_type' => 'required|string',
        'payment_type' => 'required|string',
        'date' => 'nullable|date',
        'customer_id' => 'nullable|exists:customers,id',
        'cash_customer_name' => 'nullable|string',
        'new_customer_name' => 'nullable|string',
        'new_customer_phone' => 'nullable|string',
        'new_customer_address' => 'nullable|string',
        'price_type' => 'nullable|string',
    ]);

    $customerId = null;
    $customerName = null;

    // Handle customer creation/selection for credit payments
    if ($request->payment_type === 'credit') {
        if ($request->filled('customer_id')) {
            $customer = Customer::find($request->customer_id);
            $customerId = $customer->id;
            $customerName = $customer->name;

            $creditDetail = CreditDetail::where('customer_id', $customerId)->first();
             if ($creditDetail) {
                $creditDetail->due_amount += $request->grand_total;
                $creditDetail->save();
            } else {
                CreditDetail::create([
                    'customer_id' => $customerId,
                    'due_amount' => $request->grand_total,
                    'remarks' => $request->remarks,
                    'is_paid'=>'0',
                ]);
            }
        } elseif ($request->filled('new_customer_name') && $request->filled('new_customer_phone')) {
            $customer = Customer::create([
                'name' => $request->new_customer_name,
                'phone' => $request->new_customer_phone,
                'address' => $request->new_customer_address,
            ]);

            $customerId = $customer->id;
            $customerName = $customer->name;

            CreditDetail::create([
                'customer_id' => $customerId,
                'due_amount' => $request->grand_total,
                'is_paid' => '0',
            ]);
        } else {
            return redirect()->back()->withErrors(['customer' => 'Customer information is required for credit payment.']);
        }
    } else {
        $customerName = $request->cash_customer_name ?? 'Cash Customer';
    }

    // Create the bill
    $bill = Bill::create([
        'bill_number' => 'BILL-' . uniqid(),
        'customer_id' => $customerId,
        'customer_name' => $customerName,
        'print_type' => $request->print_type,
        'remarks' => $request->remarks ?? null,
        'total_amount' => $request->grand_total,
        'payment' => $request->payment_type,
        'price_type' => $request->price_type,
        'date' => $request->date ?: now(),
    ]);

    // Store bill items
    foreach ($request->products as $productData) {
        $product = Product::find($productData['id']);

        if (!$product) {
            return redirect()->back()->withErrors(['product' => 'Product not found.']);
        }

        // Initialize variables for bill item creation
        $qty = $productData['qty'];
        $price = $productData['price'] ?? $productData['price_per_piece'];
        $priceDiff = $productData['price_diff'] ?? null;
        $amount = $productData['total'];
        $soldAs = $productData['sold_as'] ?? 'packet';

        // Handle Loss Pack scenario
       if ($soldAs === 'loss_pack' && isset($productData['loss_pack_quantity'])) {
    $lossPackQty = (int) $productData['loss_pack_quantity'];

    if ($product->pieces_per_packet >= $lossPackQty) {
        $product->pieces_per_packet -= $lossPackQty;

        // If pieces_per_packet becomes 0, reduce packet quantity by 1
        if ($product->pieces_per_packet === 0) {
            if ($product->quantity > 0) {
                $product->quantity -= 1;

                // Optionally reset pieces_per_packet (e.g. assume default per packet is 10)
                $product->pieces_per_packet = $product->default_pieces_per_packet;
            } else {
                return redirect()->back()->withErrors([
                    'stock' => "No packets left to subtract for product '{$product->name}' after using all pieces.",
                ]);
            }
        }

        $qty = $lossPackQty;
        $priceDiff = null; // No price difference for Loss Pack
    } else {
        return redirect()->back()->withErrors([
            'stock' => "Not enough pieces in packet for product '{$product->name}'. Only {$product->pieces_per_packet} pieces available, but {$lossPackQty} requested."
        ]);
    }
}


        // Create the bill item
        BillItem::create([
            'bill_id' => $bill->id,
            'product_id' => $productData['id'],
            'sale' => $request->sale_type,
            'qty' => $qty,
            'price' => $price,
            'sold_as'=>$soldAs,
            'retail_price' => $productData['retail_price'],
            'less_from_retail' => $priceDiff,
            'amount' => $amount,
            'created_at' => now()->timezone('Asia/Karachi'),
            'updated_at' => now()->timezone('Asia/Karachi'),
        ]);

        // Update product availability
        if ($soldAs === 'packet') {
            $product->quantity -= $qty;
            if ($product->quantity === 0) {
                $product->availability = 'Out-Of-Stock';
            } elseif ($product->quantity < 10) {
                $product->availability = 'Low-Stock';
            } else {
                $product->availability = 'In-Stock';
            }
        }

        $product->save();
    }

    return redirect()->route('print.view', $bill->id)->with('success', 'Bill submitted successfully!');
}


public function showBill(Request $request)
{
    $status = $request->input('status', 'completed');
    $search = $request->input('search');
    $searchType = $request->input('search_type', 'all');

    $bills = Bill::with('billItems.product')
                ->where('status', $status)
                ->when($search, function($query) use ($search, $searchType) {
                    return $query->where(function($q) use ($search, $searchType) {
                        switch ($searchType) {
                            case 'customer':
                                $q->where('customer_name', 'like', "%{$search}%");
                                break;
                            case 'bill_number':
                                $q->where('bill_number', 'like', "%{$search}%");
                                break;
                            case 'bill_id':
                                $q->where('id', $search);
                                break;
                            case 'product':
                                $q->whereHas('billItems.product', function($productQuery) use ($search) {
                                    $productQuery->where('name', 'like', "%{$search}%");
                                });
                                break;
                            default:
                                $q->where('customer_name', 'like', "%{$search}%")
                                  ->orWhere('bill_number', 'like', "%{$search}%")
                                  ->orWhere('id', $search)
                                  ->orWhereHas('billItems.product', function($productQuery) use ($search) {
                                      $productQuery->where('name', 'like', "%{$search}%");
                                  });
                        }
                    });
                })
                ->orderBy('created_at', 'desc')
                ->paginate(15); // Adjust pagination as needed

    return view('bill.bill-details', compact('bills', 'status'));
}
public function edit(Bill $bill)
{
    // Only allow editing pending bills
    if ($bill->status !== 'pending') {
        abort(403, 'Only pending bills can be edited');
    }

    return view('bill.edit', compact('bill'));
}

public function update(Request $request, Bill $bill)
{
    // Validate the request
    $validated = $request->validate([
        'products' => 'required|array',
        'products.*.id' => 'required|exists:products,id',
        'products.*.qty' => 'required|numeric|min:1',
        'products.*.price' => 'required|numeric|min:0',
        'print_type' => 'required|string',
        'remarks' => 'nullable|string',
        'grand_total' => 'required|numeric',
        // 'sale_type' => 'required|string',
        'date' => 'nullable|date',
        'price_type' => 'required|in:doctor,purchase,local',
    ]);

    // Start a database transaction
    DB::beginTransaction();

    try {
        // Update bill information
        $billData = [
            'print_type' => $request->print_type,
            'remarks' => $request->remarks,
            'total_amount' => $request->grand_total,
            'status' => 'completed', // Mark as completed after edit
            'price_type' => $request->price_type,
            'date' => $request->date ?: now(), // Use selected date or current date
        ];

        $bill->update($billData);

        // Rest of your update logic remains the same...
        foreach ($request->products as $productData) {
            $product = Product::find($productData['id']);
            
            $billItem = BillItem::where('bill_id', $bill->id)
                              ->where('product_id', $productData['id'])
                              ->first();
            
            if ($billItem) {
                $qtyDifference = $productData['qty'] - $billItem->qty;
                
                $billItem->update([
                    'qty' => $productData['qty'],
                    'price' => $productData['price'],
                    'amount' => $productData['price'] * $productData['qty'],
                    'less_from_retail' => $product->retail_price - $productData['price'],
                ]);
                
                $product->quantity -= $qtyDifference;
                $product->save();
            }
        }

        DB::commit();
        
        return redirect()->route('print.view', $bill->id)
                         ->with('success', 'Bill updated successfully!');
    } catch (\Exception $e) {
        DB::rollBack();
        return back()->withErrors(['error' => 'Failed to update bill: ' . $e->getMessage()]);
    }
}
public function printView($id)
{
    
    $bill = Bill::with('billItems.product')->findOrFail($id);
 $printSetting = PrintSetting::first();
    return view('bill.print-view', compact('bill', 'printSetting'));
}

public function makePending($id)
{
    $bill = Bill::findOrFail($id);

    // Only proceed if bill is not already pending
    if ($bill->status !== 'pending') {
        // Store the bill total before zeroing it
        $billTotal = $bill->total_amount;
        
        // Process product stock updates
        foreach ($bill->billItems as $item) {
            $product = $item->product;

            if ($product) {
                 $product->quantity += $item->qty;

                // Update availability status
                if ($product->quantity === 0 && $product->pieces_per_packet === 0) {
                    $product->availability = 'Out-Of-Stock';
                } elseif ($product->quantity < 10) {
                    $product->availability = 'Low-Stock';
                } else {
                    $product->availability = 'In-Stock';
                }

                $product->save();

                // Zero out the bill item values
                $item->amount = 0;
                $item->qty = 0;
                $item->save();
            } else {
                \Log::warning("Product not found for bill item ID: {$item->id}");
            }
        }

        // Zero out the bill total
        $bill->total_amount = 0;
        $bill->status = 'pending';
        $bill->save();

        // Process customer credit adjustment if customer exists and payment was credit
        if ($bill->customer_id && $bill->payment === 'credit') {
            $creditDetail = CreditDetail::where('customer_id', $bill->customer_id)->first();
            
            if ($creditDetail) {
                // Subtract the original bill total from due payment
                $creditDetail->due_amount -= $billTotal;
                $creditDetail->save();
}
                // Create a transaction record for audit
               else {
                \Log::warning("Credit detail not found for customer ID: {$bill->customer_id}");
            }
        }
    }

    return redirect()->route('bill.details')->with('success', 'Bill marked as pending and all transactions reversed successfully.');
}



public function markComplete( $id)
{
    //dd($request->all());
    $bill = Bill::findOrFail($id);

    if ($bill->status === 'pending') {
        foreach ($bill->billItems as $item) {
            $product = $item->product;

            if ($product) {
                
                    // Regular packet sale
                    $product->quantity -= $item->qty;

                // Update availability status
                if ($product->quantity === 0 && $product->pieces_per_packet === 0) {
                    $product->availability = 'Out-Of-Stock';
                } elseif ($product->quantity < 10) {
                    $product->availability = 'Low-Stock';
                } else {
                    $product->availability = 'In-Stock';
                }

                $product->save();
            } else {
                \Log::warning("Product not found for bill item ID: {$item->id}");
            }
        }

        $bill->status = 'completed';
        $bill->save();

        // Process customer credit if payment was credit
        if ($bill->customer_id && $bill->payment === 'credit') {
            $creditDetail = CreditDetail::where('customer_id', $bill->customer_id)->first();
            
            if ($creditDetail) {
                // Add the bill total to due payment
                $creditDetail->due_amount += $bill->total_amount;
                $creditDetail->save();

                
            } else {
                \Log::warning("Credit detail not found for customer ID: {$bill->customer_id}");
            }
        }
    }

    return response()->json(['success' => true]);
}

public function showPendingBills()
{
    $bills = Bill::with('billItems.product')
                 ->where('status', 'pending')
                 ->get();

    return view('bill.', compact('bills'));
}
public function showCredit()
{

    $credits = CreditDetail::with('customer')->get();

    return view('bill.credit.showCredit', compact('credits'));
}
public function searchCredit(Request $request)
{
    $query = CreditDetail::with('customer');

    if ($search = $request->input('search')) {
        $query->whereHas('customer', function ($q) use ($search) {
            $q->where('name', 'like', "%{$search}%")
              ->orWhere('phone', 'like', "%{$search}%");
        });
    }

    $credits = $query->get();

    return view('bill.credit.showCredit', compact('credits'));
}
public function creditCreate()
{
    return view('bill.credit.createCredit');
}
public function creditStore(Request $request)
{
    //dd($request->all());
    $validated = $request->validate([
        'name' => 'nullable|string',
        'amount' => 'nullable|string',
        'phone' => 'nullable|string',
        'address' => 'nullable|string',
    ]);
    $customer = Customer::create([
        'name' => $request->name,
        'phone' => $request->phone,
        'address' => $request->address,
    ]);
    CreditDetail::create([
        'customer_id' => $customer->id,
        'due_amount' => $request->amount,
        'is_paid'=>'0',
    ]);
    return redirect()->route('credit.show')->with('success');
}
public function creditUpdate(Request $request, $id)
{
    $validated = $request->validate([
        'due_amount' => 'required|numeric|min:0',
    ]);

    $credit = CreditDetail::findOrFail($id);

    // Subtract the paid amount from the due amount
    $paidAmount = $validated['due_amount'];
    $credit->due_amount = max(0, $credit->due_amount - $paidAmount);

    // Add the paid amount to is_paid
    $credit->is_paid += $paidAmount;

    $credit->save();

    // Create CreditAdjustment record
    $now = Carbon::now('Asia/Karachi');

    CreditAdjustment::create([
        'credit_id'   => $credit->id,
        'paid_amount' => $paidAmount,
        'due_amount'  => $credit->due_amount,
        'date'        => $now->toDateString(),
        'time'        => $now->toTimeString(),
    ]);

    return redirect()->back()->with('success', 'Due amount updated successfully.');
}
public function creditDelete($id)
{
    $credit = CreditDetail::findOrFail($id);
    $credit->delete();

    return redirect()->route('credit.show')->with('success', 'Product deleted successfully.');
}
public function addDue(Request $request, CreditDetail $credit)
{
    $request->validate([
        'add_amount' => 'required|numeric|min:1'
    ]);

    $credit->due_amount += $request->add_amount;
    $credit->save();
     $now = Carbon::now('Asia/Karachi');

    CreditAdjustment::create([
        'credit_id'   => $credit->id,
        'paid_amount' => 0,
        'due_amount'  => $request->add_amount,
        'date'        => $now->toDateString(),
        'time'        => $now->toTimeString(),
    ]);

    return redirect()->back()->with('success', 'Due amount added successfully!');
}
public function showCoustomerCredit($customer_id)
{
    $customer = Customer::findOrFail($customer_id);

    $bills = Bill::with('billItems.product')
        ->where('customer_id', $customer_id)
        ->orderBy('created_at', 'asc')
        ->get();

    // Get one credit detail per customer
    $credit = CreditDetail::where('customer_id', $customer_id)->first();

    $adjustments = CreditAdjustment::where('credit_id', $credit->id)
        ->orderBy('date', 'asc')
        ->orderBy('time', 'asc')
        ->get();

    $records = collect();
    $runningBalance = 0;
    $totalPaid = 0;
    $totalDue = 0;

    foreach ($bills as $bill) {
        $amount = $bill->billItems->sum('amount');
        $runningBalance -= $amount;
        $paid=0;
        $totalDue += $amount;

        $records->push([
        'type' => 'bill',
        'date' => $bill->created_at,
        'time' => $bill->created_at->setTimezone('Asia/Karachi'), // Set timezone here
        'amount' => $amount,
        'balance' => $runningBalance,
        'data' => $bill,
    ]);
}

foreach ($adjustments as $adjustment) {
    $dateTime = Carbon::parse($adjustment->date . ' ' . $adjustment->time)->setTimezone('Asia/Karachi');
    $runningBalance += $adjustment->paid_amount;
    $totalDue += $adjustment->due_amount;

    $records->push([
        'type' => 'payment',
        'date' => $dateTime,
        'time' => $dateTime, // Use the same datetime object
        'amount' => $adjustment->paid_amount,
        'balance' => $adjustment->due_amount,
        'data' => $adjustment,
    ]);
}

    $sortedRecords = $records->sortBy('date');
    $groupedRecords = $sortedRecords->groupBy(function ($item) {
        return $item['date']->setTimezone('Asia/Karachi')->format('Y-m-d');
    });

    return view('bill.credit.customerCredit', [
        'customer' => $customer,
        'records' => $groupedRecords,
        'currentBalance' => $runningBalance,
        'paid' => $credit->is_paid ?? 0,
        'due' => $credit->due_amount ?? 0,
        'total' => $credit->total ?? 0,
    ]);
}

public function liveSearchSuggestions(Request $request)
{
    $query = $request->get('query');
    $results = [];

    if ($query) {
        // Get bills where customer_name or bill_number or id matches
        $bills = \App\Models\Bill::where('customer_name', 'like', "%{$query}%")
            ->orWhere('id', 'like', "%{$query}%")
            ->orWhere('bill_number', 'like', "%{$query}%")
            ->limit(10)
            ->get();

        // Add customer name suggestions (unique)
        foreach ($bills as $bill) {
            $results[] = " {$bill->customer_name}";
        }

        // Get product names from bill items matching the query
        $products = Product::where('name', 'like', "%{$query}%")
            ->limit(10)
            ->pluck('name')->unique();

        // Add product name suggestions
        foreach ($products as $productName) {
            $results[] = " {$productName}";
        }

        // Remove duplicates
        $results = array_unique($results);
    }

    return response()->json(array_values($results));
}
public function returnliveSearchSuggestions(Request $request)
{
    $query = $request->get('query');
    $results = [];

    if ($query) {
        // Get bills where customer_name or bill_number or id matches
        $bills = \App\Models\Bill::where('customer_name', 'like', "%{$query}%")
            ->orWhere('id', 'like', "%{$query}%")
            ->orWhere('bill_number', 'like', "%{$query}%")
            ->limit(10)
            ->get();

        // Add customer name suggestions (unique)
        foreach ($bills as $bill) {
            $results[] = " {$bill->customer_name}";
        }

        // Get product names from bill items matching the query
        $products = Product::where('name', 'like', "%{$query}%")
            ->limit(10)
            ->pluck('name')->unique();

        // Add product name suggestions
        foreach ($products as $productName) {
            $results[] = " {$productName}";
        }

        // Remove duplicates
        $results = array_unique($results);
    }

    return response()->json(array_values($results));
}  
public function OutsideBill(Request $request)
{
    //dd($request->all());
    DB::beginTransaction();
    
    try {
        // Validate the request
        $validated = $request->validate([
            'customer_name' => 'required|string|max:255',
            'payment_type' => 'required|in:cash,card,online',
            'total_amount' => 'required|numeric|min:0',
            'products' => 'required|array|min:1',
            'products.*.name' => 'required|string|max:255',
            'products.*.type' => 'required|string|max:255',
            'products.*.qty' => 'required|integer|min:1',
            'products.*.price' => 'required|numeric|min:0',
        ]);
        
        // Create the bill
        $bill = Bill::create([
            'bill_number' => 'BILL-' . uniqid(),
            'customer_name' => $validated['customer_name'],
            'payment' => $validated['payment_type'],
            'total_amount' => $validated['total_amount'],
            'date' => Carbon::now(),
            'price_type' => 'local',
            'print_type' => 'by_name',
            'sale'       => 'outside',
            'status' => 'completed',
        ]);
        
        // Create outside items for each product
        foreach ($validated['products'] as $product) {
            OutsideItem::create([
                'bill_id' => $bill->id,
                'product_name' => $product['name'],
                'type' => $product['type'],
                'qty' => $product['qty'],
                'price' => $product['price'],
            ]);
        }
        
        DB::commit();
        
        return redirect()->route('print.outside', $bill->id)->with('success', 'Bill submitted successfully!');
        
    } catch (\Exception $e) {
        DB::rollBack();
        return response()->json([
            'success' => false,
            'message' => 'Failed to create order: ' . $e->getMessage()
        ], 500);
    
    }
}
public function printOutside($id)
{
    $bill = Bill::with('OutsideItems')->findOrFail($id);
 $printSetting = PrintSetting::first();
    return view('bill.print-outside', compact('bill', 'printSetting'));
}
// BillController.php
public function showOutsideBills(Request $request)
{
    $status = $request->input('status', 'completed');
    $search = $request->input('search');
    $searchType = $request->input('search_type', 'all');

    $query = OutsideItem::with('bill') // Eager load the bill relationship
        ->when($search, function($query) use ($search, $searchType) {
            return $query->where(function($q) use ($search, $searchType) {
                switch ($searchType) {
                    case 'customer':
                        $q->whereHas('bill', function($billQuery) use ($search) {
                            $billQuery->where('customer_name', 'like', "%{$search}%");
                        });
                        break;
                    case 'bill_number':
                        $q->whereHas('bill', function($billQuery) use ($search) {
                            $billQuery->where('bill_number', 'like', "%{$search}%");
                        });
                        break;
                    case 'bill_id':
                        $q->whereHas('bill', function($billQuery) use ($search) {
                            $billQuery->where('id', $search);
                        });
                        break;
                    case 'product':
                        $q->where('product_name', 'like', "%{$search}%");
                        break;
                    default:
                        $q->where('product_name', 'like', "%{$search}%")
                          ->orWhereHas('bill', function($billQuery) use ($search) {
                              $billQuery->where('customer_name', 'like', "%{$search}%")
                                        ->orWhere('bill_number', 'like', "%{$search}%")
                                        ->orWhere('id', $search);
                          });
                }
            });
        })
        ->whereHas('bill', function($query) use ($status) {
            $query->where('status', $status);
        })
        ->orderBy('created_at', 'desc');

    $outsideItems = $query->paginate(15);

    return view('bill.outside-bill', compact('outsideItems', 'status'));
}
}
