<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\DB;
use App\Models\SupplierCompany;
use App\Models\Supplier;
use App\Models\Company;
use App\Models\Bill;
use App\Models\BillItem;
use App\Models\PrintSetting;
use App\Models\LedgerItem;
use App\Models\Product;
use App\Models\CreditDetail;
use App\Models\SupplierLedger;
use App\Models\LedgerAdjustment;
use App\Models\ExtraExpense;
use Illuminate\Http\Request;
use Carbon\Carbon;
class SupplierController extends Controller
{
    public function index()
    {
$suppliers=Supplier::all();
return view('supplier.index', compact('suppliers'));
    }   
    public function supplierForm()
{
    return view('supplier.create');
}
    public function store(Request $request)
    {
        $request->validate([
            'supplier_name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
        ]);

        Supplier::create($request->all());

        return redirect()->route('supplier.show')->with('success');
    }

    // Update
    public function update(Request $request, $id)
    {
        $supplier = Supplier::findOrFail($id);

        $request->validate([
            'supplier_name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
        ]);

        $supplier->update($request->all());

        return redirect()->route('supplier.show')->with('success');;
    }

    // Delete
    public function destroy($id)
    {
        $supplier = Supplier::findOrFail($id);
        $supplier->delete();

        return redirect()->route('supplier.show')->with('success');
    }
    public function ShowLedger(Request $request)
{
    $query = SupplierLedger::with(['company', 'supplier']);

    // Filter by search
    if ($request->filled('search')) {
        $searchTerm = $request->input('search');
        $query->whereHas('supplier', function ($q) use ($searchTerm) {
            $q->where('supplier_name', 'like', "%{$searchTerm}%")
              ->orWhere('phone', 'like', "%{$searchTerm}%");
        });
    }

    // Filter by company
    if ($request->filled('company_filter')) {
        $query->where('company_id', $request->input('company_filter'));
    }

    $ledgers = $query->orderBy('id', 'desc')->paginate(20);

    // Get companies for filter dropdown
    $companies = Company::all();

    return view('supplier.ledger.index', compact('ledgers', 'companies'));
}
     public function ledgerDetails($id)
{
    $ledger = SupplierLedger::with(['company', 'supplier', 'ledgerItems'])
                ->findOrFail($id);
                
    // Get all manual adjustments for this ledger
    $adjustments = \App\Models\LedgerAdjustment::where('ledger_id', $id)
        ->orderBy('date', 'desc')
        ->orderBy('time', 'desc')
        ->get();
                
    return view('supplier.ledger.details', compact('ledger', 'adjustments'));
}
    public function ShowLedgerForm()
    {
        $companies=Company::all();
        $suppliers=Supplier::all();
        return view('supplier.ledger.create', compact('companies','suppliers'));
    }
  public function ledgerStore(Request $request)
{
    $request->validate([
        'supplier_id' => 'required|exists:suppliers,id',
        'company_id' => 'required|exists:companies,id',
        'products' => 'required|string',
        'total_amount' => 'required|numeric',
        'paid_amount' => 'required|numeric',
        'ledger_date' => 'required|date',
        'payment_date' => 'nullable|date',
        'image' => 'nullable|image|max:2048',
    ]);

    // Process image upload
    $imagePath = null;
    if ($request->hasFile('image')) {
        $imagePath = $request->file('image')->store('ledgers', 'public');
    }

    $dueAmount = $request->total_amount - $request->paid_amount;

    // Check if a ledger already exists for the supplier
    $supplierLedger = SupplierLedger::where('supplier_id', $request->supplier_id)->first();

    if (!$supplierLedger) {
        // Create new SupplierLedger with full details
        $supplierLedger = SupplierLedger::create([
            'supplier_id' => $request->supplier_id,
            'company_id' => $request->company_id,
            'total_bill' => $request->total_amount,
            'paid_amount' => $request->paid_amount,
            'due_amount' => $dueAmount,
            'image' => $imagePath,
        ]);
    } else {
        // Update existing SupplierLedger totals
        $supplierLedger->update([
            'total_bill' => $supplierLedger->total_bill + $request->total_amount,
            'paid_amount' => $supplierLedger->paid_amount + $request->paid_amount,
            'due_amount' => $supplierLedger->due_amount + $dueAmount,
            'image' => $imagePath ?? $supplierLedger->image, // update image only if new provided
        ]);
    }

    // Create the LedgerItem
    LedgerItem::create([
        'ledger_id' => $supplierLedger->id,
        'stock' => $request->products,
        'total_amount' => $request->total_amount,
        'paid_amount' => $request->paid_amount,
        'ledger_date' => $request->ledger_date,
        'payment_date' => $request->payment_date,
    ]);

    // Update or create the pivot entry
    DB::table('company_supplier')->updateOrInsert([
        'supplier_id' => $request->supplier_id,
        'company_id' => $request->company_id,
    ]);

    return redirect()->route('ledger.show')->with('success', 'Ledger entry saved successfully.');
}

public function ledgerUpdate(Request $request, $id)
{
    $request->validate([
        'new_paid' => 'required|numeric|min:0',
    ]);

    $ledger = SupplierLedger::findOrFail($id);

    $newPaid = $request->input('new_paid');

    $ledger->paid_amount += $newPaid;
    $ledger->due_amount = $ledger->total_bill - $ledger->paid_amount;

    $ledger->save();
    $now = Carbon::now('Asia/Karachi');

    LedgerAdjustment::create([
        'ledger_id'   => $ledger->id,
        'paid_amount' =>$request->input('new_paid'),
        'due_amount'  => $ledger->due_amount,
        'date'        => $now->toDateString(),
        'time'        => $now->toTimeString(),
    ]);

    return redirect()->back()->with('success', 'Ledger updated successfully.');
}

public function ledgerdestroy($id)
{
    $ledger = SupplierLedger::findOrFail($id);
    $ledger->delete();

    return redirect()->back()->with('success', 'Record deleted successfully.');
}
public function cashOutReport(Request $request)
{
    // Supplier Payments Query
    $supplierQuery = SupplierLedger::with(['supplier', 'company']);

    // Extra Expenses Query
    $expenseQuery = ExtraExpense::query();

    // Apply date filter if provided
    if ($request->has('start_date') && $request->has('end_date')) {
        $startDate = Carbon::parse($request->start_date)->startOfDay();
        $endDate = Carbon::parse($request->end_date)->endOfDay();
        
        $supplierQuery->whereBetween('updated_at', [$startDate, $endDate]);
        $expenseQuery->whereBetween('expense_date', [$startDate, $endDate]);
    } else {
        // Default to today if no dates provided
        $supplierQuery->whereDate('updated_at', Carbon::today());
        $expenseQuery->whereDate('expense_date', Carbon::today());
    }

    // Get formatted data
    $supplierPayments = $supplierQuery->get()
        ->map(function ($ledger) {
            return [
                'supplier_name' => $ledger->supplier->supplier_name ?? 'N/A',
                'company_name' => $ledger->company->company_name ?? 'N/A',
                'product_name' => $ledger->products ?? 'N/A',
                'total_bill' => $ledger->total_bill,
                'paid_amount' => $ledger->paid_amount,
                'due_amount' => $ledger->due_amount,
                'stock_date' => $ledger->stock_date,
            ];
        });

    $extraExpenses = $expenseQuery->get()
        ->map(function ($expense) {
            return [
                'amount' => $expense->amount,
                'description' => $expense->description,
                'expense_date' => $expense->expense_date,
            ];
        });

    return view('supplier.cash-out', compact('supplierPayments', 'extraExpenses'));
}
public function creditReport(Request $request)
{
    // Supplier Payments Query
    $supplierQuery = SupplierLedger::with(['supplier', 'company']);

    // Extra Expenses Query
    $creditQuery = CreditDetail::with('customer');

    // Apply date filter if provided
    if ($request->has('start_date') && $request->has('end_date')) {
        $startDate = Carbon::parse($request->start_date)->startOfDay();
        $endDate = Carbon::parse($request->end_date)->endOfDay();
        
        $supplierQuery->whereBetween('updated_at', [$startDate, $endDate]);
        $creditQuery->whereBetween('updated_at', [$startDate, $endDate]);
    } else {
        // Default to today if no dates provided
        $supplierQuery->whereDate('updated_at', Carbon::today());
        $creditQuery->whereDate('updated_at', Carbon::today());
    }

    // Get formatted data
    $supplierPayments = $supplierQuery->get()
        ->map(function ($ledger) {
            return [
                'supplier_name' => $ledger->supplier->supplier_name ?? 'N/A',
                'company_name' => $ledger->company->company_name ?? 'N/A',
                'product_name' => $ledger->products ?? 'N/A',
                'total_bill' => $ledger->total_bill,
                'paid_amount' => $ledger->paid_amount,
                'due_amount' => $ledger->due_amount,
                'stock_date' => $ledger->stock_date,
            ];
        });

    $creditDetail = $creditQuery->get()
        ->map(function ($credit) {
            return [
                'customer_name' => $credit->customer->name,
                'phone' => $credit->customer->phone,
                'due_amount' => $credit->due_amount,
                'is_paid' => $credit->is_paid,
                'date' => $credit->updated_at,
            ];
        });

    return view('bill.credit.creditDatetoDate', compact('supplierPayments', 'creditDetail'));
}
public function profitReport(Request $request)
{
    $query = BillItem::with(['bill', 'product'])
        ->join('bills', 'bill_items.bill_id', '=', 'bills.id')
        ->join('products', 'bill_items.product_id', '=', 'products.id')
        ->where('bill_items.amount', '>', 0)  // Only include items with positive amount
        ->where('bill_items.qty', '>', 0)     // Only include items with positive quantity
        ->where('bills.sale', '!=', 'outside'); // Exclude outside sales

    // Apply date filter if provided
    if ($request->has('start_date') && $request->has('end_date')) {
        $startDate = Carbon::parse($request->start_date)->startOfDay();
        $endDate = Carbon::parse($request->end_date)->endOfDay();
        $query->whereBetween('bills.date', [$startDate, $endDate]);
    } else {
        $query->whereDate('bills.date', Carbon::today());
    }

    $sales = $query->select(
        'bill_items.*',
        'products.purchase_price',
        'products.retail_price',
        'products.local_price',
        'bills.date as sale_date',
        'bills.sale' // Include sale column in selection
    )->get();

    $profitData = $sales->map(function ($item) {
        // Calculate profit based on your business logic
        $costPrice = $item->purchase_price * $item->qty;
        $sellingPrice = $item->amount;
        $profit = $sellingPrice - $costPrice;
        $profitMargin = ($sellingPrice > 0) ? ($profit / $sellingPrice) * 100 : 0;

        return [
            'bill_id' => $item->bill_id,
            'product_name' => $item->product->name,
            'quantity' => $item->qty,
            'purchase_price' => $item->purchase_price,
            'selling_price' => $item->price,
            'total_selling' => $item->amount,
            'total_cost' => $costPrice,
            'profit' => $profit,
            'profit_margin' => $profitMargin,
            'sale_date' => $item->sale_date,
            'payment_method' => $item->bill->payment,
            'sale_type' => $item->bill->sale // Include sale type in the output
        ];
    });

    // Summary calculations
    $totalSales = $profitData->sum('total_selling');
    $totalCost = $profitData->sum('total_cost');
    $totalProfit = $totalSales - $totalCost;
    $averageMargin = ($totalSales > 0) ? ($totalProfit / $totalSales) * 100 : 0;

    return view('profitReport', compact('profitData', 'totalSales', 'totalCost', 'totalProfit', 'averageMargin'));
}
public function creditPrint($id)
{
     $printSetting = PrintSetting::first();
    $credit = CreditDetail::with('customer')->findOrFail($id);
    return view('bill.credit.print', compact('credit','printSetting'));
}

}
