@extends('layouts.app')

@section('content')
<div class="max-w-7xl mx-auto p-4 sm:p-6 bg-white rounded-xl shadow-md mt-6 border border-gray-200  top-0  right-0 z-50 margin-top" s>
   <div class="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-4 gap-4">
        <h2 class="text-xl sm:text-2xl font-bold text-gray-800">Product Records</h2>
        <div class="flex flex-col sm:flex-row gap-3 w-full sm:w-auto">
            <a href="{{ route('product.create') }}" class="bg-blue-600 hover:bg-blue-700 text-white font-semibold px-4 py-2 rounded text-sm sm:text-base text-center">
                Add Product
            </a>
            <a href="{{ route('product.print') }}" class="bg-purple-600 hover:bg-purple-700 text-white font-semibold px-4 py-2 rounded text-sm sm:text-base text-center shadow">
                Print All
            </a>
        </div>
    </div>
    
    <form method="GET" action="{{ route('product.index', $filter) }}" class="mb-4">
    <div class="flex flex-col sm:flex-row items-stretch sm:items-center gap-2 w-full">
        <div class="relative flex-grow sm:max-w-md w-full">
            <input type="text" name="search" id="liveSearchInput"
                   value="{{ request('search') }}"
                   placeholder="Search by name or company..."
                   oninput="handleLiveSearch(this.value)"
                   autocomplete="off"
                   class="w-full px-4 py-2 border border-gray-300 rounded shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-300 text-sm">
            <!-- Live search suggestions -->
            <div id="liveResultContainer"
                 class="absolute top-full left-0 w-full bg-white border border-gray-300 rounded shadow text-sm z-10 hidden">
            </div>
        </div>
        <button type="submit"
                class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded text-sm">
            Search
        </button>
    </div>
   </form>

</div>
@if (session('success'))
    <div class="mb-3 p-2 bg-green-100 text-green-700 rounded text-sm">
        {{ session('success') }}
    </div>
@endif

@if ($errors->any())
    <div class="mb-3 p-2 bg-red-100 text-red-700 rounded text-sm">
        <ul class="list-disc ml-4">
            @foreach ($errors->all() as $error)
                <li>{{ $error }}</li>
            @endforeach
        </ul>
    </div>
@endif

@if($products->count())
<table class="min-w-full border-separate border-spacing-0">
    <thead class="bg-gray-100 text-black-700 font-semibold text-sm uppercase top-0 z-10 fixed ">
        <tr>
            <th class="py-1 px-2 border-b">Product Name</th>
            <th class="py-1 px-2 border-b">Categories</th>
            <th class="py-1 px-2 border-b">Company</th>
            <th class="py-1 px-2 border-b">Batch No</th>
            <th class="py-1 px-2 border-b">Expiry Date</th>
            <th class="py-1 px-2 border-b">Qty</th>
            <th class="py-1 px-2 border-b">Unit</th>
            <th class="py-1 px-2 border-b">Purchase</th>
            <th class="py-1 px-2 border-b">Retail</th>
            <th class="py-1 px-2 border-b">Local</th>
            <th class="py-1 px-2 border-b">Doctor</th>
            
            <th class="py-1 px-2 border-b">Image</th>
            <th class="py-1 px-2 border-b">Actions</th>
        </tr>
    </thead>
</table>
<div class="max-w-7xl mx-auto p-4 sm:p-6 bg-white rounded-xl shadow-md mt-4 border border-gray-200 ">
    <div class="overflow-x-auto text-xs sm:text-sm">

        <div class="max-h-[500px] overflow-y-auto">
            <table class="min-w-full table-fixed border-separate border-spacing-0">
                <tbody>
                @foreach ($products as $product)
                <tr class="border-b @if($product->is_near_expiry) bg-purple-100 @elseif($product->is_low_stock) bg-red-100 @elseif($product->is_expired) bg-yellow-100 @endif hover:bg-gray-50 transition duration-200">
                    <td class="py-1 px-2">{{ $product->name }}</td>
                    <td class="py-1 px-2">
                        <div class="flex flex-wrap gap-1">
                            @foreach ($product->categories as $category)
                                <span class="bg-gray-200 text-gray-800 px-1 py-0.5 rounded">{{ $category }}</span>
                            @endforeach
                        </div>
                    </td>
                    <td class="py-1 px-2">{{ $product->company }}</td>
                    <td class="py-1 px-2">{{ $product->batch_number }}</td>
                    <td class="py-1 px-2">{{ $product->expiry_date }}</td>
                    <td class="py-1 px-2 font-medium @if($product->quantity == 0) text-red-600 @elseif($product->is_low_stock) text-orange-600 @endif">
                        {{ $product->quantity }}
                    </td>
                    <td class="py-1 px-2">{{ $product->unit_type }}</td>
                    <td class="py-1 px-2">{{ $product->purchase_price }}</td>
                    <td class="py-1 px-2">{{ $product->retail_price }}</td>
                    <td class="py-1 px-2">{{ $product->local_price }}</td>
                    <td class="py-1 px-2">{{ $product->doctor_price }}</td>
                    
                    <td class="py-1 px-2">
                        @if ($product->image)
                            <img src="{{ asset('storage/' . $product->image) }}" alt="Product Image"
                                 class="w-8 h-8 rounded shadow border border-gray-300 object-cover">
                        @else
                            <span class="text-gray-400">No Image</span>
                        @endif
                    </td>
                    <td class="py-1 px-2">
                        <div class="flex flex-col gap-1">
                            <a href="{{ route('product.edit', $product->id) }}"
                               class="bg-green-500 hover:bg-green-600 text-white px-1.5 py-0.5 rounded text-xs font-semibold text-center">
                                Edit
                            </a>
                            <form action="{{ route('product.delete', $product->id) }}" method="POST" onsubmit="return confirm('Delete {{ $product->name }}?')">
                                @csrf
                                @method('DELETE')
                                <button type="submit"
                                        class="w-full bg-red-500 hover:bg-red-600 text-white px-1.5 py-0.5 rounded text-xs font-semibold">
                                    Delete
                                </button>
                            </form>
                            <a href="{{ route('products.sales', $product->id) }}"
                               class="bg-blue-500 hover:bg-blue-600 text-white px-1.5 py-0.5 rounded text-xs font-semibold text-center">
                                Sales
                            </a>
                            @if($product->quantity == 0)
                                <button onclick="addToDemand({{ $product->id }}, '{{ $product->name }}')"
                                   class="w-full bg-yellow-500 hover:bg-orange-600 text-white px-1.5 py-0.5 rounded text-xs font-semibold">
                                   Demand
                                </button>
                            @endif
                        </div>
                    </td>
                </tr>
                @endforeach
                </tbody>
            </table>
        </div>
    </div>

    <div class="mt-4 flex flex-col sm:flex-row items-center justify-center gap-3 text-xs sm:text-sm text-gray-600">
        <div class="flex items-center gap-1">
            <span class="inline-block w-3 h-3 bg-purple-100 border border-gray-300 rounded"></span>
            <span>Near Expiry</span>
        </div>
        <div class="flex items-center gap-1">
            <span class="inline-block w-3 h-3 bg-yellow-100 border border-gray-300 rounded"></span>
            <span> Expired</span>
        </div>
        <div class="flex items-center gap-1">
            <span class="inline-block w-3 h-3 bg-red-100 border border-gray-300 rounded"></span>
            <span>Low Stock & Out of Stock</span>
        </div>
       
    </div>

    @else
        <p class="text-gray-600 mt-3 text-sm">No products found.</p>
    @endif
</div>

<script>
function addToDemand(productId, productName) {
    const qty = prompt(`Enter quantity for ${productName}:`, 20);
    
    if(qty && !isNaN(qty) && qty > 0) {
        // Show loading indicator
        const button = event.target;
        button.disabled = true;
        button.innerHTML = 'Adding...';
        
        fetch(`/demands/add-from-product/${productId}`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Accept': 'application/json'
            },
            body: JSON.stringify({
                product_name: productName,
                qty: qty
            })
        })
        .then(response => {
            if (!response.ok) {
                return response.json().then(err => { throw err; });
            }
            return response.json();
        })
        .then(data => {
            alert(data.message || 'Product added to demand successfully!');
            window.location.reload();
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error: ' + (error.message || 'Failed to add to demand. Please try again.'));
        })
        .finally(() => {
            button.disabled = false;
            button.innerHTML = 'Demand';
        });
    } else if(qty !== null) {
        alert('Please enter a valid quantity (must be a number greater than 0)');
    }
}
let selectedIndex = -1;
let currentSuggestions = [];

function handleLiveSearch(query) {
    const container = document.getElementById('liveResultContainer');

    if (!query.trim()) {
        container.classList.add('hidden');
        container.innerHTML = '';
        currentSuggestions = [];
        selectedIndex = -1;
        return;
    }

    fetch(`{{ route('product.searchStarts') }}?q=${encodeURIComponent(query)}`, {
        headers: { 'Accept': 'application/json' }
    })
    .then(res => res.json())
    .then(data => {
        currentSuggestions = data;
        selectedIndex = -1;

        if (data.length === 0) {
            container.innerHTML = `<div class="px-3 py-2 text-gray-500">No matching products</div>`;
        } else {
            container.innerHTML = data.map((product, index) => `
                <div class="px-3 py-2 suggestion-item hover:bg-blue-100 cursor-pointer border-b border-gray-200"
                     data-index="${index}"
                     onclick="selectSuggestion('${product.name.replace(/'/g, "\\'")}')">
                    <strong>${product.name}</strong> <span class="text-gray-500">(${product.company})</span>
                </div>
            `).join('');
        }
        container.classList.remove('hidden');
    })
    .catch(err => {
        console.error(err);
        container.classList.add('hidden');
    });
}

function selectSuggestion(name) {
    const input = document.getElementById('liveSearchInput');
    input.value = name;

    const container = document.getElementById('liveResultContainer');
    container.classList.add('hidden');
    container.innerHTML = '';

    input.focus();
    // Optionally submit form: input.form.submit();
}

function updateActiveItem(items) {
    items.forEach((item, index) => {
        if (index === selectedIndex) {
            item.classList.add('bg-blue-100');
            item.scrollIntoView({ block: 'nearest' });
        } else {
            item.classList.remove('bg-blue-100');
        }
    });
}

document.addEventListener('DOMContentLoaded', function () {
    const input = document.getElementById('liveSearchInput');

    input.addEventListener('keydown', function (e) {
        const container = document.getElementById('liveResultContainer');
        const items = container.querySelectorAll('[data-index]');

        if (container.classList.contains('hidden') || items.length === 0) return;

        if (e.key === 'ArrowDown') {
            e.preventDefault();
            selectedIndex = (selectedIndex + 1) % items.length;
            document.body.classList.add('using-keyboard');
            updateActiveItem(items);
        } else if (e.key === 'ArrowUp') {
            e.preventDefault();
            selectedIndex = (selectedIndex - 1 + items.length) % items.length;
            document.body.classList.add('using-keyboard');
            updateActiveItem(items);
        } else if (e.key === 'Enter' && selectedIndex >= 0) {
            e.preventDefault();
            const selectedProduct = currentSuggestions[selectedIndex];
            selectSuggestion(selectedProduct.name);
        }
    });

    // Optional: reset selectedIndex if input is clicked/focused again
    input.addEventListener('focus', () => {
        selectedIndex = -1;
    });
});

// Hide suggestions if clicked outside
document.addEventListener('click', function (e) {
    const input = document.getElementById('liveSearchInput');
    const container = document.getElementById('liveResultContainer');
    if (!input.contains(e.target) && !container.contains(e.target)) {
        container.classList.add('hidden');
    }
});

// Handle hover/keyboard interaction conflict
let usingKeyboard = false;
document.addEventListener('keydown', (e) => {
    if (['ArrowUp', 'ArrowDown'].includes(e.key)) {
        usingKeyboard = true;
        document.body.classList.add('using-keyboard');
    }
});
document.addEventListener('mousemove', () => {
    if (usingKeyboard) {
        usingKeyboard = false;
        document.body.classList.remove('using-keyboard');
    }
});
</script>
<style>
    .margin-top{
        margin-top: 5%;
    }
    body.using-keyboard #liveResultContainer .suggestion-item:hover {
    background-color: inherit !important;
}
</style>
@endsection